<?php
declare(strict_types=1);
error_reporting(E_ALL);
ini_set('display_errors', '1');

session_start();
require_once '../includes/db_connect.php';
require_once '../includes/functions.php';

// Check if the user is logged in as an admin
if (!is_admin()) {
    header("Location: login.php");
    exit();
}

$stats = [
    'total_products' => 0,
    'total_orders' => 0,
    'total_revenue' => 0.00,
    'recent_orders' => [],
    'pending_orders' => [],
    'daily_orders' => [],
    'recent_activities' => []
];

$errors = [];
$success_message = '';

// Function to safely execute a query and handle errors
function safe_query($conn, $query, $error_message) {
    global $errors;
    $result = $conn->query($query);
    if ($result === false) {
        $errors[] = $error_message . ": " . $conn->error;
        return null;
    }
    return $result;
}

// Handle EA upload
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['upload_ea'])) {
    // ... [EA upload code remains unchanged] ...
}

// Fetch statistics
try {
    // Total products
    $result = safe_query($conn, "SELECT COUNT(*) as product_count FROM products", "Error counting products");
    if ($result && $row = $result->fetch_assoc()) {
        $stats['total_products'] = (int)$row['product_count'];
    }

    // Total orders and revenue
    $result = safe_query($conn, "SELECT COUNT(*) as order_count, COALESCE(SUM(total_amount), 0) as revenue FROM orders", "Error calculating order stats");
    if ($result && $row = $result->fetch_assoc()) {
        $stats['total_orders'] = (int)$row['order_count'];
        $stats['total_revenue'] = (float)$row['revenue'];
    }

    // Recent orders
    $result = safe_query($conn, "SELECT id, total_amount, status, created_at FROM orders ORDER BY created_at DESC LIMIT 5", "Error fetching recent orders");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $row['total_amount'] = (float)$row['total_amount'];  // Ensure total_amount is treated as float
            $stats['recent_orders'][] = $row;
        }
    }

    // Pending orders
    $result = safe_query($conn, "SELECT id, total_amount, created_at FROM orders WHERE status = 'pending' ORDER BY created_at ASC LIMIT 5", "Error fetching pending orders");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $row['total_amount'] = (float)$row['total_amount'];  // Ensure total_amount is treated as float
            $stats['pending_orders'][] = $row;
        }
    }

    // Daily orders for the past week
    $result = safe_query($conn, "SELECT DATE(created_at) as date, COUNT(*) as count FROM orders WHERE created_at >= DATE_SUB(CURDATE(), INTERVAL 7 DAY) GROUP BY DATE(created_at)", "Error calculating daily orders");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $stats['daily_orders'][$row['date']] = (int)$row['count'];
        }
    }

    // Recent activities
    $result = safe_query($conn, "SELECT action, created_at FROM activity_log ORDER BY created_at DESC LIMIT 10", "Error fetching recent activities");
    if ($result) {
        while ($row = $result->fetch_assoc()) {
            $stats['recent_activities'][] = $row;
        }
    }

} catch (Exception $e) {
    $errors[] = "Unexpected error: " . $e->getMessage();
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
</head>
<body>
    <div class="container mt-4">
        <h1>Admin Dashboard</h1>
        
        <?php if (!empty($errors)): ?>
            <div class="alert alert-danger">
                <h4>Errors Occurred:</h4>
                <ul>
                    <?php foreach ($errors as $error): ?>
                        <li><?= htmlspecialchars($error) ?></li>
                    <?php endforeach; ?>
                </ul>
            </div>
        <?php endif; ?>

        <?php if (!empty($success_message)): ?>
            <div class="alert alert-success">
                <?= htmlspecialchars($success_message) ?>
            </div>
        <?php endif; ?>

        <div class="row mt-4">
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Total Products</h5>
                        <p class="card-text display-4"><?= $stats['total_products'] ?></p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Total Orders</h5>
                        <p class="card-text display-4"><?= $stats['total_orders'] ?></p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Total Revenue</h5>
                        <p class="card-text display-4">$<?= number_format($stats['total_revenue'], 2) ?></p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Pending Orders</h5>
                        <p class="card-text display-4"><?= count($stats['pending_orders']) ?></p>
                    </div>
                </div>
            </div>
        </div>

        <div class="row mt-4">
            <div class="col-md-6">
                <h2>Upload New EA</h2>
                <form action="" method="post" enctype="multipart/form-data">
                    <div class="mb-3">
                        <label for="ea_name" class="form-label">EA Name</label>
                        <input type="text" class="form-control" id="ea_name" name="ea_name" required>
                    </div>
                    <div class="mb-3">
                        <label for="ea_price" class="form-label">Price</label>
                        <input type="number" class="form-control" id="ea_price" name="ea_price" step="0.01" required>
                    </div>
                    <div class="mb-3">
                        <label for="ea_description" class="form-label">Description</label>
                        <textarea class="form-control" id="ea_description" name="ea_description" rows="3"></textarea>
                    </div>
                    <div class="mb-3">
                        <label for="ea_file" class="form-label">EA File</label>
                        <input type="file" class="form-control" id="ea_file" name="ea_file" required>
                    </div>
                    <div class="mb-3">
                        <label for="image_file" class="form-label">Featured Image</label>
                        <input type="file" class="form-control" id="image_file" name="image_file" accept="image/*">
                    </div>
                    <button type="submit" name="upload_ea" class="btn btn-primary">Upload EA</button>
                </form>
            </div>
            <div class="col-md-6">
                <h2>Recent Orders</h2>
                <table class="table table-striped">
                    <thead>
                        <tr>
                            <th>Order ID</th>
                            <th>Amount</th>
                            <th>Status</th>
                            <th>Date</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($stats['recent_orders'] as $order): ?>
                            <tr>
                                <td><?= htmlspecialchars($order['id']) ?></td>
                                <td>$<?= number_format($order['total_amount'], 2) ?></td>
                                <td><?= htmlspecialchars($order['status']) ?></td>
                                <td><?= date('M j, Y', strtotime($order['created_at'])) ?></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
        </div>

        <div class="row mt-4">
            <div class="col-md-6">
                <h2>Daily Orders (Past Week)</h2>
                <canvas id="dailyOrdersChart"></canvas>
            </div>
            <div class="col-md-6">
                <h2>Recent Activities</h2>
                <ul class="list-group">
                    <?php foreach ($stats['recent_activities'] as $activity): ?>
                        <li class="list-group-item">
                            <?= htmlspecialchars($activity['action']) ?>
                            <small class="float-end text-muted"><?= date('M j, Y H:i', strtotime($activity['created_at'])) ?></small>
                        </li>
                    <?php endforeach; ?>
                </ul>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Chart for daily orders
        var ctx = document.getElementById('dailyOrdersChart').getContext('2d');
        var chart = new Chart(ctx, {
            type: 'line',
            data: {
                labels: <?= json_encode(array_keys($stats['daily_orders'])) ?>,
                datasets: [{
                    label: 'Daily Orders',
                    data: <?= json_encode(array_values($stats['daily_orders'])) ?>,
                    borderColor: 'rgb(75, 192, 192)',
                    tension: 0.1
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true
                    }
                }
            }
        });
    </script>
</body>
</html>
<?php
$conn->close();
?>