<?php
// Enable error reporting for debugging
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

session_start();
require_once 'includes/db_connect.php';

// Redirect if cart is empty
if (empty($_SESSION['cart'])) {
    header("Location: cart.php");
    exit();
}

// Function to get exchange rate
function getExchangeRate($from, $to) {
    $url = "https://open.er-api.com/v6/latest/{$from}";
    
    $curl = curl_init();
    curl_setopt($curl, CURLOPT_URL, $url);
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
    
    $response = curl_exec($curl);
    
    if ($response === false) {
        error_log("cURL Error: " . curl_error($curl));
        return null;
    }
    
    curl_close($curl);
    
    $data = json_decode($response, true);
    
    if (isset($data['rates'][$to])) {
        return $data['rates'][$to];
    } else {
        error_log("Exchange rate not found in API response");
        return null;
    }
}

// Function to generate PayFast signature
function generateSignature($data, $passPhrase = null) {
    $pfOutput = '';
    foreach ($data as $key => $val) {
        if ($val !== '') {
            $pfOutput .= $key . '=' . urlencode(trim($val)) . '&';
        }
    }
    $getString = substr($pfOutput, 0, -1);
    if ($passPhrase !== null) {
        $getString .= '&passphrase=' . urlencode(trim($passPhrase));
    }
    return md5($getString);
}

$errors = [];
$success_message = '';

// Get USD to ZAR exchange rate
$usd_to_zar_rate = getExchangeRate('USD', 'ZAR');
if (!$usd_to_zar_rate) {
    $errors[] = "Unable to fetch current exchange rate. Using fallback rate of 18.5.";
    $usd_to_zar_rate = 18.5; // Fallback rate, update this periodically
}

// Calculate cart total in USD for display
$cart_total_usd = 0;
foreach ($_SESSION['cart'] as $product_id => $quantity) {
    $stmt = $conn->prepare("SELECT price FROM products WHERE id = ?");
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $stmt->bind_result($price);
    if ($stmt->fetch()) {
        $cart_total_usd += $price * $quantity;
    }
    $stmt->close();
}

// Handle form submission for payment
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Collect customer details
    $name = $_POST['name'] ?? '';
    $email = $_POST['email'] ?? '';
    $contact = $_POST['contact'] ?? '';

    // Validate customer details
    if (empty($name) || empty($email) || empty($contact)) {
        $errors[] = "All fields are required.";
    } else {
        try {
            // Insert customer into the database
            $stmt = $conn->prepare("INSERT INTO customers (name, email, contact) VALUES (?, ?, ?)");
            $stmt->bind_param("sss", $name, $email, $contact);
            if (!$stmt->execute()) {
                throw new Exception("Error saving customer: " . $stmt->error);
            }
            $customer_id = $stmt->insert_id;
            $stmt->close();

            // Convert total to ZAR
            $total_zar = $cart_total_usd * $usd_to_zar_rate;

            // Set PayFast credentials and endpoint
            $payfast_merchant_id = '10000100';
            $payfast_merchant_key = '46f0cd694581a';
            $payfast_passphrase = 'jt7NOE43FZPn';
            $payfast_url = 'https://sandbox.payfast.co.za/eng/process';

            // Prepare PayFast request data
            $payfast_data = [
                'merchant_id' => $payfast_merchant_id,
                'merchant_key' => $payfast_merchant_key,
                'return_url' => 'https://online.mticon.com/success.php',
                'cancel_url' => 'https://online.mticon.com/cancel.php',
                'notify_url' => 'https://online.mticon.com/notify.php',
                'name_first' => $name,
                'email_address' => $email,
                'm_payment_id' => $customer_id,
                'amount' => number_format($total_zar, 2, '.', ''),
                'item_name' => 'Order Payment',
                'email_confirmation' => 1,
                'confirmation_address' => $email
            ];

            // Generate PayFast signature
            $signature = generateSignature($payfast_data, $payfast_passphrase);
            $payfast_data['signature'] = $signature;

            // Log the PayFast request data (remove in production)
            error_log("PayFast Request Data: " . print_r($payfast_data, true));
            error_log("Generated Signature: " . $signature);

            // Redirect to PayFast payment page
            $query_string = http_build_query($payfast_data);
            $payfast_url = $payfast_url . '?' . $query_string;
            
            // Log the full PayFast URL (remove in production)
            error_log("PayFast URL: " . $payfast_url);

            header('Location: ' . $payfast_url);
            exit();

        } catch (Exception $e) {
            error_log("Checkout Error: " . $e->getMessage());
            $errors[] = "An error occurred while processing your order. Please try again.";
        }
    }
}

include 'includes/header.php';
?>

<div class="container mt-5">
    <h1>Checkout</h1>
    <?php if (!empty($errors)): ?>
        <div class="alert alert-danger">
            <ul>
                <?php foreach ($errors as $error): ?>
                    <li><?= htmlspecialchars($error) ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php endif; ?>

    <?php if (!empty($success_message)): ?>
        <div class="alert alert-success">
            <?= htmlspecialchars($success_message) ?>
        </div>
    <?php endif; ?>

    <div class="alert alert-info">
        <p>Order Total: $<?= number_format($cart_total_usd, 2) ?> USD</p>
        <p>Approximate Total in ZAR: R<?= number_format($cart_total_usd * $usd_to_zar_rate, 2) ?> ZAR</p>
        <p><small>Exchange Rate: 1 USD = <?= number_format($usd_to_zar_rate, 4) ?> ZAR</small></p>
        <?php if ($usd_to_zar_rate == 18.5): ?>
            <p><small>Note: Using estimated exchange rate. Actual rate may vary.</small></p>
        <?php endif; ?>
    </div>

    <form action="checkout.php" method="post">
        <div class="mb-3">
            <label for="name" class="form-label">Full Name</label>
            <input type="text" class="form-control" id="name" name="name" required>
        </div>
        <div class="mb-3">
            <label for="email" class="form-label">Email Address</label>
            <input type="email" class="form-control" id="email" name="email" required>
        </div>
        <div class="mb-3">
            <label for="contact" class="form-label">Contact Number</label>
            <input type="text" class="form-control" id="contact" name="contact" required>
        </div>
        <button type="submit" class="btn btn-primary">Proceed to Pay</button>
    </form>
</div>

<?php
include 'includes/footer.php';
?>