<?php
session_start();
require_once 'includes/db_connect.php';

// Redirect to cart if it's empty
if (empty($_SESSION['cart'])) {
    header("Location: cart.php");
    exit();
}

$pageTitle = "Checkout - Forex EA Store";
include 'includes/header.php';

// Process the order
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Validate form data (add more validation as needed)
    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_STRING);
    $email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
    $address = filter_input(INPUT_POST, 'address', FILTER_SANITIZE_STRING);
    
    if ($name && $email && $address) {
        // Create the order
        $conn->begin_transaction();
        
        try {
            // Insert order into the database
            $sql = "INSERT INTO orders (customer_name, customer_email, customer_address, total_amount) VALUES (?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            
            // Calculate total amount
            $total_amount = 0;
            foreach ($_SESSION['cart'] as $product_id => $quantity) {
                $product_sql = "SELECT price FROM products WHERE id = ?";
                $product_stmt = $conn->prepare($product_sql);
                $product_stmt->bind_param("i", $product_id);
                $product_stmt->execute();
                $product_result = $product_stmt->get_result();
                $product = $product_result->fetch_assoc();
                $total_amount += $product['price'] * $quantity;
                $product_stmt->close();
            }
            
            $stmt->bind_param("sssd", $name, $email, $address, $total_amount);
            $stmt->execute();
            $order_id = $stmt->insert_id;
            $stmt->close();
            
            // Insert order items
            $sql = "INSERT INTO order_items (order_id, product_id, quantity, price) VALUES (?, ?, ?, ?)";
            $stmt = $conn->prepare($sql);
            
            foreach ($_SESSION['cart'] as $product_id => $quantity) {
                $product_sql = "SELECT price FROM products WHERE id = ?";
                $product_stmt = $conn->prepare($product_sql);
                $product_stmt->bind_param("i", $product_id);
                $product_stmt->execute();
                $product_result = $product_stmt->get_result();
                $product = $product_result->fetch_assoc();
                
                $stmt->bind_param("iiid", $order_id, $product_id, $quantity, $product['price']);
                $stmt->execute();
                
                $product_stmt->close();
            }
            
            $stmt->close();
            
            // Commit the transaction
            $conn->commit();
            
            // Clear the cart
            unset($_SESSION['cart']);
            
            // Send confirmation email (implement this function)
            // send_confirmation_email($email, $order_id);
            
            // Redirect to a thank you page
            header("Location: thank-you.php?order_id=" . $order_id);
            exit();
        } catch (Exception $e) {
            // Rollback the transaction if an error occurred
            $conn->rollback();
            $error_message = "An error occurred while processing your order. Please try again.";
        }
    } else {
        $error_message = "Please fill in all required fields.";
    }
}
?>

<div class="container mt-5">
    <h1>Checkout</h1>
    <?php if (isset($error_message)): ?>
        <div class="alert alert-danger"><?php echo $error_message; ?></div>
    <?php endif; ?>
    <form action="checkout.php" method="post">
        <div class="mb-3">
            <label for="name" class="form-label">Full Name</label>
            <input type="text" class="form-control" id="name" name="name" required>
        </div>
        <div class="mb-3">
            <label for="email" class="form-label">Email Address</label>
            <input type="email" class="form-control" id="email" name="email" required>
        </div>
        <div class="mb-3">
            <label for="address" class="form-label">Address</label>
            <textarea class="form-control" id="address" name="address" required></textarea>
        </div>
        <h2 class="mt-4">Order Summary</h2>
        <table class="table">
            <thead>
                <tr>
                    <th>Product</th>
                    <th>Quantity</th>
                    <th>Price</th>
                    <th>Subtotal</th>
                </tr>
            </thead>
            <tbody>
                <?php
                $total = 0;
                foreach ($_SESSION['cart'] as $product_id => $quantity):
                    $sql = "SELECT name, price FROM products WHERE id = ?";
                    $stmt = $conn->prepare($sql);
                    $stmt->bind_param("i", $product_id);
                    $stmt->execute();
                    $result = $stmt->get_result();
                    $product = $result->fetch_assoc();
                    $subtotal = $product['price'] * $quantity;
                    $total += $subtotal;
                ?>
                    <tr>
                        <td><?php echo htmlspecialchars($product['name']); ?></td>
                        <td><?php echo $quantity; ?></td>
                        <td>$<?php echo number_format($product['price'], 2); ?></td>
                        <td>$<?php echo number_format($subtotal, 2); ?></td>
                    </tr>
                <?php
                    $stmt->close();
                endforeach;
                ?>
            </tbody>
            <tfoot>
                <tr>
                    <th colspan="3">Total</th>
                    <th>$<?php echo number_format($total, 2); ?></th>
                </tr>
            </tfoot>
        </table>
        <div class="text-end mt-4">
            <button type="submit" class="btn btn-primary btn-lg">Place Order</button>
        </div>
    </form>
</div>

<?php
$conn->close();
include 'includes/footer.php';
?>