<?php
// Start session if not already started
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

/**
 * Sanitize user input
 * 
 * @param string $input The input to sanitize
 * @return string The sanitized input
 */
function sanitize_input($input) {
    return htmlspecialchars(strip_tags(trim($input)));
}

/**
 * Check if user is logged in
 * 
 * @return bool True if user is logged in, false otherwise
 */
function is_logged_in() {
    return isset($_SESSION['user_id']);
}

/**
 * Check if user is an admin
 * 
 * @return bool True if user is an admin, false otherwise
 */
function is_admin() {
    return isset($_SESSION['admin_id']);
}

/**
 * Redirect user to a specific page
 * 
 * @param string $location The URL to redirect to
 */
function redirect($location) {
    header("Location: $location");
    exit();
}

/**
 * Generate CSRF token
 * 
 * @return string The generated CSRF token
 */
function generate_csrf_token() {
    if (!isset($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF token
 * 
 * @param string $token The token to verify
 * @return bool True if token is valid, false otherwise
 */
function verify_csrf_token($token) {
    return isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * Add item to cart
 * 
 * @param int $product_id The ID of the product to add
 * @param int $quantity The quantity to add (default 1)
 */
function add_to_cart($product_id, $quantity = 1) {
    if (!isset($_SESSION['cart'])) {
        $_SESSION['cart'] = [];
    }
    
    if (isset($_SESSION['cart'][$product_id])) {
        $_SESSION['cart'][$product_id] += $quantity;
    } else {
        $_SESSION['cart'][$product_id] = $quantity;
    }
}

/**
 * Remove item from cart
 * 
 * @param int $product_id The ID of the product to remove
 */
function remove_from_cart($product_id) {
    if (isset($_SESSION['cart'][$product_id])) {
        unset($_SESSION['cart'][$product_id]);
    }
}

/**
 * Get cart total
 * 
 * @return float The total price of items in the cart
 */
function get_cart_total() {
    global $conn;
    $total = 0;
    
    if (isset($_SESSION['cart']) && !empty($_SESSION['cart'])) {
        $product_ids = array_keys($_SESSION['cart']);
        $placeholders = implode(',', array_fill(0, count($product_ids), '?'));
        
        $sql = "SELECT id, price FROM products WHERE id IN ($placeholders)";
        $stmt = $conn->prepare($sql);
        $stmt->bind_param(str_repeat('i', count($product_ids)), ...$product_ids);
        $stmt->execute();
        $result = $stmt->get_result();
        
        while ($row = $result->fetch_assoc()) {
            $total += $row['price'] * $_SESSION['cart'][$row['id']];
        }
        
        $stmt->close();
    }
    
    return $total;
}

/**
 * Format price
 * 
 * @param float $price The price to format
 * @return string The formatted price
 */
function format_price($price) {
    return '$' . number_format($price, 2);
}

/**
 * Get product details
 * 
 * @param int $product_id The ID of the product
 * @return array|null The product details or null if not found
 */
function get_product_details($product_id) {
    global $conn;
    
    $sql = "SELECT * FROM products WHERE id = ?";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("i", $product_id);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        return null;
    }
    
    $product = $result->fetch_assoc();
    $stmt->close();
    
    return $product;
}

/**
 * Log an action (for admin purposes)
 * 
 * @param string $action The action to log
 * @param string $details Additional details about the action
 */
function log_action($action, $details = '') {
    global $conn;
    
    $user_id = $_SESSION['user_id'] ?? 0;
    $sql = "INSERT INTO activity_log (user_id, action, details) VALUES (?, ?, ?)";
    $stmt = $conn->prepare($sql);
    $stmt->bind_param("iss", $user_id, $action, $details);
    $stmt->execute();
    $stmt->close();
}

/**
 * Send email
 * 
 * @param string $to Recipient email address
 * @param string $subject Email subject
 * @param string $message Email body
 * @return bool True if email was sent successfully, false otherwise
 */
function send_email($to, $subject, $message) {
    // You might want to use a library like PHPMailer for more robust email functionality
    $headers = "From: noreply@yourstore.com\r\n";
    $headers .= "Reply-To: support@yourstore.com\r\n";
    $headers .= "X-Mailer: PHP/" . phpversion();
    
    return mail($to, $subject, $message, $headers);
}

/**
 * Generate a unique order number
 * 
 * @return string The generated order number
 */
function generate_order_number() {
    return 'ORD-' . strtoupper(uniqid());
}

/**
 * Check if a string is a valid JSON
 * 
 * @param string $string The string to check
 * @return bool True if the string is valid JSON, false otherwise
 */
function is_valid_json($string) {
    json_decode($string);
    return (json_last_error() == JSON_ERROR_NONE);
}

?>